<?php

namespace Tests\Feature;

use App\Models\User;
use App\Models\LeaveType;
use App\Models\StaffProfile;
use App\Models\LeaveRequest;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;
use Spatie\Permission\Models\Role;

class LeaveSystemTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        // Setup Roles
        Role::create(['name' => 'Admin']);
        Role::create(['name' => 'Staff']);
    }

    public function test_admin_can_create_leave_type()
    {
        $admin = User::factory()->create();
        $admin->assignRole('Admin');

        $response = $this->actingAs($admin)->postJson('/api/leaves/types', [
            'name' => 'Sick Leave',
            'is_paid' => true,
            'deduction_percentage' => 0,
        ]);

        $response->assertStatus(201);
        $this->assertDatabaseHas('leave_types', ['name' => 'Sick Leave']);
    }

    public function test_user_can_create_leave_request()
    {
        $user = User::factory()->create();
        $user->assignRole('Staff');

        $response = $this->actingAs($user)->postJson('/api/leaves/requests', [
            'start_date' => '2025-01-01',
            'end_date' => '2025-01-03', // 3 days
            'reason' => 'Flu',
        ]);

        $response->assertStatus(201);
        $this->assertDatabaseHas('leave_requests', [
            'user_id' => $user->id,
            'reason' => 'Flu'
        ]);
    }

    public function test_admin_can_approve_and_deduct_quota()
    {
        $admin = User::factory()->create();
        $admin->assignRole('Admin');

        $department = \App\Models\Department::create(['name' => 'IT', 'description' => 'IT Dept']);

        $user = User::factory()->create();
        $user->assignRole('Staff');
        StaffProfile::create([
            'user_id' => $user->id,
            'department_id' => $department->id,
            'monthly_compensation' => 50000,
            'total_paid_leaves' => 12,
            'used_paid_leaves' => 0,
        ]);

        $leaveType = LeaveType::create([
            'name' => 'Casual Leave',
            'is_paid' => true,
            'deduction_percentage' => 0,
        ]);

        // Assuming 2025-01-06 is Monday, 2025-01-07 is Tuesday
        $request = LeaveRequest::create([
            'user_id' => $user->id,
            'start_date' => '2025-01-06', 
            'end_date' => '2025-01-07', 
            'reason' => 'Trip',
            'status' => 'pending',
        ]);

        $response = $this->actingAs($admin)->putJson("/api/leaves/requests/{$request->id}/status", [
            'status' => 'approved',
            'leave_type_id' => $leaveType->id,
        ]);

        $response->assertStatus(200);
        $this->assertEquals('approved', $request->fresh()->status);
        $this->assertEquals(2, $user->staffProfile->fresh()->used_paid_leaves);
    }
}
