<?php

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\AttendanceController;
use App\Http\Controllers\PayrollController;
use App\Http\Controllers\TaskController;
use App\Http\Controllers\StaffController;
use App\Http\Controllers\DashboardController;
use App\Http\Controllers\NotificationController;
use App\Http\Controllers\StockController;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use App\Models\User;
use App\Services\AttendanceService;
use App\Http\Controllers\LeaveTypeController;
use App\Http\Controllers\LeaveRequestController;
use App\Http\Controllers\Api\ReimbursementController;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for your application. These
| routes are loaded by the RouteServiceProvider within a group which
| is assigned the "api" middleware group. Enjoy building your API!
|
*/

// Authentication Routes (Public)
Route::post('/login', function (Request $request) {
    $request->validate([
        'email' => 'required|email',
        'password' => 'required',
    ]);

    $user = User::where('email', $request->email)->first();

    if (!$user || !Hash::check($request->password, $user->password)) {
        throw ValidationException::withMessages([
            'email' => ['The provided credentials are incorrect.'],
        ]);
    }

    $token = $user->createToken('api-token')->plainTextToken;

    // Check and update attendance after successful login
    try {
        // Always check and update attendance for all staff (service handles filtering)
        $attendanceService = app(AttendanceService::class);
        
        // Temporarily authenticate the user for attendance service
        Auth::login($user);
        
        $updated = $attendanceService->checkAndUpdateAttendance();
        
        if ($updated) {
            Log::info('Attendance auto-update check completed after API login by user: ' . $user->name);
        }
        
        // Logout after attendance check to maintain stateless API
        Auth::logout();
    } catch (\Exception $e) {
        // Log error but don't block the login process
        Log::error('Error checking attendance after API login: ' . $e->getMessage());
    }

    // Get today's attendance data for the user
    $todayAttendance = null;
    $requiresSelfie = false;
    
    try {
        $today = \Carbon\Carbon::today();
        $todayAttendance = \App\Models\Attendance::where('user_id', $user->id)
                                                ->whereDate('date', $today)
                                                ->first();
        
        // Determine if selfie is required based on attendance type and existing attendance
        if ($user->attendance_type === 'check-in') {
            $requiresSelfie = !$todayAttendance; // Require selfie if no attendance for today
        }
    } catch (\Exception $e) {
        Log::error('Error fetching today\'s attendance during login: ' . $e->getMessage());
        // If there's an error and user has check-in type, require selfie to be safe
        if ($user->attendance_type === 'check-in') {
            $requiresSelfie = true;
        }
    }

    return response()->json([
        'token' => $token,
        'user' => $user->load('staffProfile'),
        'attendance_type' => $user->attendance_type,
        'today_attendance' => $todayAttendance,
        'requires_selfie' => $requiresSelfie,
        'message' => 'Login successful'
    ]);
});

// Protected Routes (Require Authentication)
Route::middleware('auth:sanctum')->group(function () {
    
    // User Authentication & Profile
    Route::post('/logout', function (Request $request) {
        // Get the current access token
        $token = $request->user()->currentAccessToken();
        
        // Check if token exists and has delete method (for API token authentication)
        if ($token && method_exists($token, 'delete')) {
            $token->delete;
        } else {
            // For session-based authentication, revoke all tokens
            $request->user()->tokens()->delete();
        }
        
        return response()->json(['message' => 'Logged out successfully']);
    });
    
    Route::get('/user', function (Request $request) {
        return response()->json($request->user()->load('staffProfile'));
    });
    
    Route::put('/user', function (Request $request) {
        $user = $request->user();
        $validated = $request->validate([
            'name' => 'sometimes|string|max:255',
            'email' => 'sometimes|email|unique:users,email,' . $user->id,
            'phone' => 'sometimes|string|max:20',
            'address' => 'sometimes|string|max:500',
        ]);
        
        $user->update($validated);
        
        return response()->json([
            'user' => $user->fresh()->load('staffProfile'),
            'message' => 'Profile updated successfully'
        ]);
    });
    
    // Dashboard
    Route::get('/dashboard', [DashboardController::class, 'apiIndex']);
    
    // Attendance Routes
    Route::prefix('attendance')->group(function () {
        Route::get('/', [AttendanceController::class, 'apiIndex']);
        Route::post('/', [AttendanceController::class, 'apiStore']);
        Route::get('/stats', [AttendanceController::class, 'apiStats']);
        Route::get('/today', [AttendanceController::class, 'apiToday']);
        Route::post('/mark', [AttendanceController::class, 'apiMarkAttendance']);
        Route::post('/checkout', [AttendanceController::class, 'apiCheckout']);
        Route::post('/selfie', [AttendanceController::class, 'apiSubmitSelfie']);
        Route::post('/debug-selfie', [AttendanceController::class, 'debugSelfieRequest']);
        Route::get('/{id}', [AttendanceController::class, 'apiShow']);
        Route::put('/{id}', [AttendanceController::class, 'apiUpdate']);
        Route::delete('/{id}', [AttendanceController::class, 'apiDestroy']);
    });
    
    // Payroll Routes
    Route::prefix('payrolls')->group(function () {
        Route::get('/', [PayrollController::class, 'apiIndex']);
        Route::post('/', [PayrollController::class, 'apiStore']);
        Route::get('/{id}', [PayrollController::class, 'apiShow']);
        Route::put('/{id}', [PayrollController::class, 'apiUpdate']);
        Route::delete('/{id}', [PayrollController::class, 'apiDestroy']);
        Route::post('/bulk', [PayrollController::class, 'apiBulkCreate']);
        Route::post('/{id}/approve', [PayrollController::class, 'apiApprove']);
        Route::post('/{id}/mark-paid', [PayrollController::class, 'apiMarkPaid']);
    });
    
    // Task Routes
    Route::prefix('tasks')->group(function () {
        Route::get('/', [TaskController::class, 'apiIndex']);
        Route::post('/', [TaskController::class, 'apiStore']);
        Route::get('/{id}', [TaskController::class, 'apiShow']);
        Route::put('/{id}', [TaskController::class, 'apiUpdate']);
        Route::delete('/{id}', [TaskController::class, 'apiDestroy']);
        Route::post('/{id}/complete', [TaskController::class, 'apiComplete']);
        Route::post('/{id}/reassign', [TaskController::class, 'apiReassign']);
        Route::get('/{id}/assignment-history', [TaskController::class, 'apiAssignmentHistory']);
        Route::post('/{id}/subtasks', [TaskController::class, 'apiCreateSubtask']);
    });
    
    // Staff Routes (Admin only)
    Route::middleware('role:Admin')->prefix('staff')->group(function () {
        Route::get('/', [StaffController::class, 'apiIndex']);
        Route::post('/', [StaffController::class, 'apiStore']);
        Route::get('/{id}', [StaffController::class, 'apiShow']);
        Route::put('/{id}', [StaffController::class, 'apiUpdate']);
        Route::delete('/{id}', [StaffController::class, 'apiDestroy']);
    });
    
    // Staff for task assignment (accessible to all authenticated users)
    Route::get('/staff-for-assignment', [StaffController::class, 'apiStaffForAssignment']);
    
    // Departments (accessible to all authenticated users)
    Route::get('/departments', function () {
        return response()->json(\App\Models\Department::select('id', 'name')->get());
    });

    // Leave Routes
    Route::prefix('leaves')->group(function () {
        // Types
        Route::get('/types', [LeaveTypeController::class, 'index']);
        Route::middleware('role:Admin')->group(function () {
            Route::post('/types', [LeaveTypeController::class, 'store']);
            Route::put('/types/{leaveType}', [LeaveTypeController::class, 'update']);
            Route::delete('/types/{leaveType}', [LeaveTypeController::class, 'destroy']);
        });

        // Requests
        Route::get('/requests', [LeaveRequestController::class, 'index']);
        Route::post('/requests', [LeaveRequestController::class, 'store']);
        
        // Admin approval
        Route::middleware('role:Admin')->group(function () {
            Route::put('/requests/{leaveRequest}/status', [LeaveRequestController::class, 'updateStatus']);
        });
    });
    
    // Notification Management
    Route::get('/notifications', [NotificationController::class, 'apiIndex']);
    Route::get('/notifications/unread-count', [NotificationController::class, 'apiUnreadCount']);
    Route::get('/notifications/persistent', [NotificationController::class, 'apiPersistent']);
    Route::get('/notifications/task-persistent', [NotificationController::class, 'apiTaskPersistent']);
    Route::put('/notifications/{id}/read', [NotificationController::class, 'apiMarkAsRead']);
    
    // Admin-only notification routes
    Route::middleware('role:Admin')->group(function () {
        Route::post('/notifications', [NotificationController::class, 'apiStore']);
        Route::post('/notifications/task-persistent', [NotificationController::class, 'apiCreateTaskPersistent']);
        Route::delete('/notifications/task-persistent/{id}', [NotificationController::class, 'apiRemoveTaskPersistent']);
        Route::post('/notifications/broadcast', [NotificationController::class, 'apiBroadcast']);
        Route::delete('/notifications/{id}', [NotificationController::class, 'apiDestroy']);
    });
    
    // Additional utility routes
    Route::get('/departments/{department}/staff', function ($departmentId) {
        return \App\Models\Department::find($departmentId)->staff;
    });
    
    Route::get('/departments/{department}/users', [TaskController::class, 'getUsersByDepartment']);
    
    // Stock Management API Routes
    Route::prefix('stock')->group(function () {
        Route::get('/', [StockController::class, 'apiIndex']);
        Route::get('/filters', [StockController::class, 'apiFilters']);
    });
});