@extends('layouts.app')

@section('page-title', 'Staff Management')

@section('content')
<div class="bg-white dark:bg-gray-800/50 backdrop-blur-md rounded-lg shadow-xl border border-gray-200 dark:border-gray-700">
    <div class="px-6 py-4 border-b border-gray-200 dark:border-gray-700">
        <div class="flex flex-col sm:flex-row sm:items-center sm:justify-between">
            <h1 class="text-2xl font-bold text-gray-900 dark:text-white mb-4 sm:mb-0">
                <i class="fas fa-users mr-3 text-red-500 dark:text-red-400"></i>
                Staff Management
            </h1>
            @can('create', App\Models\User::class)
                <div class="flex flex-wrap items-center gap-2 mt-4 sm:mt-0">
                    <a href="{{ route('staff.sample') }}" class="bg-green-600 hover:bg-green-700 text-white px-3 py-2 rounded-md text-sm font-medium transition-colors duration-200">
                        <i class="fas fa-download mr-1"></i> Sample CSV
                    </a>

                    <button onclick="openDepartmentModal()" class="bg-purple-600 hover:bg-purple-700 text-white px-3 py-2 rounded-md text-sm font-medium transition-colors duration-200">
                        <i class="fas fa-building mr-1"></i> Departments
                    </button>
                    
                    <button onclick="document.getElementById('import-form').classList.toggle('hidden')" class="bg-blue-600 hover:bg-blue-700 text-white px-3 py-2 rounded-md text-sm font-medium transition-colors duration-200">
                        <i class="fas fa-file-import mr-1"></i> Bulk Upload
                    </button>

                    <a href="{{ route('staff.create') }}" class="bg-red-600 hover:bg-red-700 text-white px-3 py-2 rounded-md text-sm font-medium transition-colors duration-200">
                        <i class="fas fa-plus mr-1"></i> Add Staff
                    </a>
                </div>
            @endcan
        </div>
    </div>

    <div id="import-form" class="hidden px-6 py-4 border-b border-gray-200 dark:border-gray-700 bg-gray-50 dark:bg-gray-700/30 transition-all duration-300">
        <form action="{{ route('staff.import') }}" method="POST" enctype="multipart/form-data" class="flex flex-col sm:flex-row items-end gap-4">
            @csrf
            <div class="w-full sm:w-auto flex-grow">
                <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">Select CSV File</label>
                <input type="file" name="file" accept=".csv" required class="block w-full text-sm text-gray-500 dark:text-gray-400
                    file:mr-4 file:py-2 file:px-4
                    file:rounded-md file:border-0
                    file:text-sm file:font-semibold
                    file:bg-red-50 file:text-red-700
                    hover:file:bg-red-100
                    dark:file:bg-red-900/30 dark:file:text-red-400
                "/>
            </div>
            <div class="w-full sm:w-auto">
                <button type="submit" class="w-full sm:w-auto bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-md text-sm font-medium transition-colors duration-200">
                    <i class="fas fa-upload mr-2"></i> Upload Data
                </button>
            </div>
        </form>
        <p class="mt-2 text-xs text-gray-500 dark:text-gray-400">
            Please ensure your CSV matches the sample format. 
        </p>

        @if(isset($departments) && $departments->count() > 0)
            <div class="mt-4 pt-4 border-t border-gray-200 dark:border-gray-600">
                <h4 class="text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">Department Reference (Use ID in CSV)</h4>
                <div class="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-6 gap-2" id="department-list-container">
                    @foreach($departments as $dept)
                        <div class="text-xs bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-600 rounded px-2 py-1">
                            <span class="font-mono font-bold text-blue-600 dark:text-blue-400">ID: {{ $dept->id }}</span>
                            <span class="text-gray-600 dark:text-gray-400 truncate block">{{ $dept->name }}</span>
                        </div>
                    @endforeach
                </div>
            </div>
        @endif
    </div>
</div> <!-- End of main content card -->

<!-- Department Management Modal (Outside the main card) -->
<div id="departmentModal" class="fixed inset-0 z-50 hidden overflow-y-auto" aria-labelledby="modal-title" role="dialog" aria-modal="true">
    <div class="flex items-end justify-center min-h-screen pt-4 px-4 pb-20 text-center sm:block sm:p-0">
        <!-- Background overlay -->
        <div class="fixed inset-0 bg-gray-500/75 transition-opacity" aria-hidden="true" onclick="closeDepartmentModal()"></div>

        <!-- This element is to trick the browser into centering the modal contents. -->
        <span class="hidden sm:inline-block sm:align-middle sm:h-screen" aria-hidden="true">&#8203;</span>

        <!-- Modal panel -->
        <div class="relative z-10 inline-block align-bottom bg-white dark:bg-gray-800 rounded-lg text-left overflow-hidden shadow-xl transform transition-all sm:my-8 sm:align-middle sm:max-w-lg w-full">
            <div class="bg-white dark:bg-gray-800 px-4 pt-5 pb-4 sm:p-6 sm:pb-4">
                <div class="sm:flex sm:items-start">
                    <div class="mx-auto flex-shrink-0 flex items-center justify-center h-12 w-12 rounded-full bg-purple-100 sm:mx-0 sm:h-10 sm:w-10">
                        <i class="fas fa-building text-purple-600"></i>
                    </div>
                    <div class="mt-3 text-center sm:mt-0 sm:ml-4 sm:text-left w-full">
                        <h3 class="text-lg leading-6 font-medium text-gray-900 dark:text-white" id="modal-title">
                            Manage Departments
                        </h3>
                        <div class="mt-2">
                            <p class="text-sm text-gray-500 dark:text-gray-400 mb-4">
                                View IDs or add new departments.
                            </p>
                            
                            <!-- Add Department Form -->
                            <form id="addDepartmentForm" class="flex gap-2 mb-4" onsubmit="event.preventDefault(); addDepartment();">
                                <input type="text" id="newDepartmentName" placeholder="New Department Name" required
                                    class="flex-grow rounded-md border-gray-300 dark:border-gray-600 shadow-sm focus:border-purple-500 focus:ring focus:ring-purple-200 dark:bg-gray-700 dark:text-white text-sm"
                                >
                                <button type="submit" class="bg-purple-600 hover:bg-purple-700 text-white px-3 py-2 rounded-md text-sm font-medium">
                                    Add
                                </button>
                            </form>

                            <!-- Department List -->
                            <div class="max-h-60 overflow-y-auto border border-gray-200 dark:border-gray-600 rounded-md">
                                <table class="min-w-full divide-y divide-gray-200 dark:divide-gray-600">
                                    <thead class="bg-gray-50 dark:bg-gray-700">
                                        <tr>
                                            <th scope="col" class="px-4 py-2 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">ID</th>
                                            <th scope="col" class="px-4 py-2 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Name</th>
                                            <th scope="col" class="px-4 py-2 text-right text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Action</th>
                                        </tr>
                                    </thead>
                                    <tbody id="modalDepartmentList" class="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-600">
                                        <!-- Populated via JS -->
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="bg-gray-50 dark:bg-gray-700/50 px-4 py-3 sm:px-6 sm:flex sm:flex-row-reverse">
                <button type="button" class="mt-3 w-full inline-flex justify-center rounded-md border border-gray-300 dark:border-gray-600 shadow-sm px-4 py-2 bg-white dark:bg-gray-800 text-base font-medium text-gray-700 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-gray-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 sm:mt-0 sm:ml-3 sm:w-auto sm:text-sm" onclick="closeDepartmentModal()">
                    Close
                </button>
            </div>
        </div>
    </div>
</div>

<script>
    function openDepartmentModal() {
        document.getElementById('departmentModal').classList.remove('hidden');
        fetchDepartments();
    }

    function closeDepartmentModal() {
        document.getElementById('departmentModal').classList.add('hidden');
    }

    function fetchDepartments() {
        fetch('{{ route("departments.index") }}', {
            headers: {
                'Accept': 'application/json',
                'X-Requested-With': 'XMLHttpRequest'
            }
        })
        .then(response => response.json())
        .then(data => {
            updateDepartmentLists(data);
        })
        .catch(error => console.error('Error fetching departments:', error));
    }

    function addDepartment() {
        const nameInput = document.getElementById('newDepartmentName');
        const name = nameInput.value;
        
        if (!name) return;

        fetch('{{ route("departments.store") }}', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json',
                'X-CSRF-TOKEN': '{{ csrf_token() }}'
            },
            body: JSON.stringify({ name: name })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                nameInput.value = '';
                fetchDepartments(); // Refresh list
                // Optional: Show success toast
            } else {
                alert(data.message || 'Error creating department');
            }
        })
        .catch(error => console.error('Error adding department:', error));
    }

    function deleteDepartment(id) {
        if (!confirm('Are you sure you want to delete this department?')) return;

        fetch(`/departments/${id}`, {
            method: 'DELETE',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json',
                'X-CSRF-TOKEN': '{{ csrf_token() }}'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                fetchDepartments(); // Refresh list
            } else {
                alert(data.message || 'Error deleting department');
            }
        })
        .catch(error => console.error('Error deleting department:', error));
    }

    function updateDepartmentLists(departments) {
        // Update Modal List
        const modalList = document.getElementById('modalDepartmentList');
        modalList.innerHTML = departments.map(dept => `
            <tr>
                <td class="px-4 py-2 whitespace-nowrap text-sm text-gray-900 dark:text-gray-300 font-mono">${dept.id}</td>
                <td class="px-4 py-2 whitespace-nowrap text-sm text-gray-900 dark:text-gray-300">${dept.name}</td>
                <td class="px-4 py-2 whitespace-nowrap text-right text-sm font-medium">
                    <button onclick="deleteDepartment(${dept.id})" class="text-red-600 hover:text-red-900 dark:hover:text-red-400">
                        <i class="fas fa-trash"></i>
                    </button>
                </td>
            </tr>
        `).join('');

        // Update Main Page Reference List (if it exists)
        const mainList = document.getElementById('department-list-container');
        if (mainList) {
            mainList.innerHTML = departments.map(dept => `
                <div class="text-xs bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-600 rounded px-2 py-1">
                    <span class="font-mono font-bold text-blue-600 dark:text-blue-400">ID: ${dept.id}</span>
                    <span class="text-gray-600 dark:text-gray-400 truncate block">${dept.name}</span>
                </div>
            `).join('');
        }
    }
</script>

<div class="p-6">
    @if (session('success'))
            <div class="mb-4 bg-green-100 dark:bg-green-900 border border-green-400 dark:border-green-700 text-green-700 dark:text-green-300 px-4 py-3 rounded-md">
                <div class="flex items-center">
                    <i class="fas fa-check-circle mr-2"></i>
                    <div>{{ session('success') }}</div>
                </div>
            </div>
        @endif

        @if (session('import_errors'))
            <div class="mb-4 bg-red-100 dark:bg-red-900/50 border border-red-400 dark:border-red-700 text-red-700 dark:text-red-300 px-4 py-3 rounded-md">
                <div class="font-medium mb-2"><i class="fas fa-exclamation-circle mr-2"></i> Import Errors:</div>
                <ul class="list-disc list-inside text-sm max-h-40 overflow-y-auto">
                    @foreach (session('import_errors') as $error)
                        <li>{{ $error }}</li>
                    @endforeach
                </ul>
            </div>
        @endif

        @if($staff->count() > 0)
            <div class="bg-gray-50 dark:bg-gray-700/50 rounded-lg border border-gray-200 dark:border-gray-600 overflow-hidden">
                <div class="overflow-x-auto">
                    <table class="w-full">
                        <thead class="bg-gray-100 dark:bg-gray-700/50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Employee ID</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Name</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Email</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Department</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Position</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Phone</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Actions</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray-200 dark:divide-gray-600 bg-white dark:bg-transparent">
                            @foreach($staff as $member)
                                <tr class="hover:bg-gray-50 dark:hover:bg-gray-700/50 transition-colors duration-200">
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-gray-300">{{ $member->staffProfile->employee_id ?? 'N/A' }}</td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">
                                        <a href="{{ route('staff.show', $member) }}" class="text-red-600 dark:text-red-400 hover:text-red-800 dark:hover:text-red-300 transition-colors duration-200">
                            {{ $member->name }}
                        </a>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-300">{{ $member->email }}</td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-300">{{ $member->staffProfile->department->name ?? 'N/A' }}</td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-300">{{ $member->staffProfile->position ?? 'N/A' }}</td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-300">{{ $member->staffProfile->phone ?? 'N/A' }}</td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                        <div class="flex space-x-2">
                                            <a href="{{ route('staff.show', $member) }}" class="bg-red-600 hover:bg-red-700 text-white px-3 py-1 rounded text-xs transition-colors duration-200">
                                <i class="fas fa-eye"></i>
                            </a>
                                            @can('update', $member)
                                                <a href="{{ route('staff.edit', $member) }}" class="bg-yellow-500 hover:bg-yellow-600 text-white px-3 py-1 rounded text-xs transition-colors duration-200">
                                                    <i class="fas fa-edit"></i>
                                                </a>
                                            @endcan
                                            @can('delete', $member)
                                                <form action="{{ route('staff.destroy', $member) }}" method="POST" class="inline" onsubmit="return confirm('Are you sure you want to delete this staff member?')">
                                                    @csrf
                                                    @method('DELETE')
                                                    <button type="submit" class="bg-red-600 hover:bg-red-700 text-white px-3 py-1 rounded text-xs transition-colors duration-200">
                                                        <i class="fas fa-trash"></i>
                                                    </button>
                                                </form>
                                            @endcan
                                        </div>
                                    </td>
                                </tr>
                            @endforeach
                        </tbody>
                        </table>
                </div>
            </div>

            <div class="mt-6 flex justify-center">
                {{ $staff->links() }}
            </div>
        @else
            <div class="text-center py-12">
                <i class="fas fa-users text-6xl text-gray-300 dark:text-gray-500 mb-4"></i>
                <h3 class="text-xl font-medium text-gray-900 dark:text-gray-300 mb-2">No staff members found</h3>
                <p class="text-gray-500 dark:text-gray-400 mb-6">Get started by adding your first staff member.</p>
                @can('create', App\Models\User::class)
                    <a href="{{ route('staff.create') }}" class="bg-red-600 hover:bg-red-700 text-white px-6 py-3 rounded-md font-medium transition-colors duration-200">
                        <i class="fas fa-plus mr-2"></i>Add First Staff Member
                    </a>
                @endcan
            </div>
        @endif
    </div>
</div>
@endsection
