@extends('layouts.app')

@section('title', 'Attendance Map View')

@section('content')
<div class="min-h-screen bg-gray-50 dark:bg-transparent py-8">
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <!-- Header -->
        <div class="mb-8">
            <div class="flex justify-between items-center">
                <div>
                    <h1 class="text-3xl font-bold text-gray-900 dark:text-white">Attendance Map View</h1>
                    <p class="mt-2 text-gray-500 dark:text-gray-400">View staff attendance locations and selfies on an interactive map</p>
                </div>
                <div class="flex space-x-4">
                    <a href="{{ route('attendance.index') }}" class="bg-white dark:bg-gray-700 hover:bg-gray-50 dark:hover:bg-gray-600 text-gray-700 dark:text-white border border-gray-300 dark:border-gray-600 px-4 py-2 rounded-lg transition-colors shadow-sm">
                        <i class="fas fa-list mr-2"></i>List View
                    </a>
                </div>
            </div>
        </div>

        <!-- Filters -->
        <div class="bg-white dark:bg-gray-800/50 backdrop-blur-md rounded-lg p-6 mb-8 shadow border border-gray-200 dark:border-gray-700">
            <form method="GET" action="{{ route('attendance.map') }}" class="grid grid-cols-1 md:grid-cols-4 gap-4">
                <div>
                    <label for="start_date" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Start Date</label>
                    <input type="date" id="start_date" name="start_date" value="{{ request('start_date') }}" 
                           class="w-full bg-white dark:bg-gray-900 border border-gray-300 dark:border-gray-600 text-gray-900 dark:text-white rounded-lg px-3 py-2 focus:ring-2 focus:ring-red-500 focus:border-transparent">
                </div>
                <div>
                    <label for="end_date" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">End Date</label>
                    <input type="date" id="end_date" name="end_date" value="{{ request('end_date') }}" 
                           class="w-full bg-white dark:bg-gray-900 border border-gray-300 dark:border-gray-600 text-gray-900 dark:text-white rounded-lg px-3 py-2 focus:ring-2 focus:ring-red-500 focus:border-transparent">
                </div>
                <div>
                    <label for="user_id" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Staff Member</label>
                    <select id="user_id" name="user_id" class="w-full bg-white dark:bg-gray-900 border border-gray-300 dark:border-gray-600 text-gray-900 dark:text-white rounded-lg px-3 py-2 focus:ring-2 focus:ring-red-500 focus:border-transparent">
                        <option value="">All Staff</option>
                        @foreach($users as $user)
                            <option value="{{ $user->id }}" {{ request('user_id') == $user->id ? 'selected' : '' }}>
                                {{ $user->name }}
                            </option>
                        @endforeach
                    </select>
                </div>
                <div class="flex items-end">
                    <button type="submit" class="w-full bg-red-600 hover:bg-red-700 text-white px-4 py-2 rounded-lg transition-colors">
                        <i class="fas fa-filter mr-2"></i>Filter
                    </button>
                </div>
            </form>
        </div>

        <!-- Map and Sidebar Layout -->
        <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
            <!-- Map Container -->
            <div class="lg:col-span-2">
                <div class="bg-white dark:bg-gray-800 rounded-lg p-4 shadow border border-gray-200 dark:border-gray-700">
                    <div id="map" class="w-full h-96 lg:h-[600px] rounded-lg"></div>
                </div>
            </div>

            <!-- Attendance List Sidebar -->
            <div class="lg:col-span-1">
                <div class="bg-white dark:bg-gray-800/50 backdrop-blur-md rounded-lg p-6 shadow border border-gray-200 dark:border-gray-700">
                    <h3 class="text-xl font-semibold text-gray-900 dark:text-white mb-4">
                        <i class="fas fa-list mr-2 text-red-500 dark:text-red-400"></i>
                        Attendance Records ({{ $attendances->count() }})
                    </h3>
                    
                    <div class="space-y-4 max-h-[500px] overflow-y-auto">
                        @forelse($attendances as $attendance)
                            <div class="attendance-item bg-gray-50 dark:bg-gray-700/50 rounded-lg p-4 cursor-pointer hover:bg-gray-100 dark:hover:bg-gray-600/50 transition-colors border border-gray-200 dark:border-gray-600" 
                                 data-lat="{{ $attendance->latitude }}" 
                                 data-lng="{{ $attendance->longitude }}"
                                 data-id="{{ $attendance->id }}">
                                
                                <!-- Staff Info -->
                                <div class="flex items-center mb-3">
                                    <div class="w-10 h-10 bg-red-600 rounded-full flex items-center justify-center text-white font-semibold mr-3">
                                        {{ substr($attendance->user->name, 0, 1) }}
                                    </div>
                                    <div>
                                        <h4 class="text-gray-900 dark:text-white font-medium">{{ $attendance->user->name }}</h4>
                                        <p class="text-gray-500 dark:text-gray-400 text-sm">{{ $attendance->date->format('M j, Y') }}</p>
                                    </div>
                                </div>

                                <!-- Status and Time -->
                                <div class="flex justify-between items-center mb-3">
                                    <span class="px-2 py-1 rounded-full text-xs font-medium
                                        @if($attendance->status === 'present') bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-300
                                        @elseif($attendance->status === 'late') bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-300
                                        @elseif($attendance->status === 'absent') bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-300
                                        @else bg-gray-100 text-gray-800 dark:bg-gray-900 dark:text-gray-300 @endif">
                                        {{ ucfirst($attendance->status) }}
                                    </span>
                                    <span class="text-gray-500 dark:text-gray-400 text-sm">
                                        {{ $attendance->check_in ? Carbon\Carbon::parse($attendance->check_in)->format('g:i A') : 'N/A' }}
                                    </span>
                                </div>

                                <!-- Location Info -->
                                @if($attendance->hasLocation())
                                    <div class="text-sm text-gray-600 dark:text-gray-300 mb-2">
                                        <i class="fas fa-map-marker-alt text-red-500 dark:text-red-400 mr-1"></i>
                                        {{ Str::limit($attendance->location_address ?: 'Location Available', 40) }}
                                    </div>
                                @endif

                                <!-- Selfie Indicator -->
                                @if($attendance->hasSelfie())
                                    <div class="flex items-center text-sm text-green-600 dark:text-green-400">
                                        <i class="fas fa-camera mr-1"></i>
                                        Selfie Available
                                    </div>
                                @endif
                            </div>
                        @empty
                            <div class="text-center py-8">
                                <i class="fas fa-map-marker-alt text-4xl text-gray-400 dark:text-gray-600 mb-4"></i>
                                <p class="text-gray-500 dark:text-gray-400">No attendance records with location data found.</p>
                            </div>
                        @endforelse
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Selfie Modal -->
<div id="selfieModal" class="fixed inset-0 bg-black bg-opacity-75 hidden z-50 flex items-center justify-center p-4">
    <div class="bg-white dark:bg-gray-800 rounded-lg max-w-2xl w-full max-h-[90vh] overflow-y-auto shadow-xl">
        <div class="p-6">
            <div class="flex justify-between items-center mb-4">
                <h3 class="text-xl font-semibold text-gray-900 dark:text-white">Attendance Details</h3>
                <button id="closeModal" class="text-gray-500 dark:text-gray-400 hover:text-gray-700 dark:hover:text-white">
                    <i class="fas fa-times text-xl"></i>
                </button>
            </div>
            
            <div id="modalContent">
                <!-- Content will be loaded dynamically -->
            </div>
        </div>
    </div>
</div>

@endsection

@push('scripts')
<!-- Leaflet CSS -->
<link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" />

<!-- Leaflet JS -->
<script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>

<script>
// Initialize map
let map;
let markers = [];
let attendanceData = @json($attendances->map(function($attendance) {
    return [
        'id' => $attendance->id,
        'user_name' => $attendance->user->name,
        'date' => $attendance->date->format('M j, Y'),
        'time' => $attendance->check_in ? \Carbon\Carbon::parse($attendance->check_in)->format('g:i A') : 'N/A',
        'status' => $attendance->status,
        'latitude' => $attendance->latitude,
        'longitude' => $attendance->longitude,
        'location_address' => $attendance->location_address,
        'has_selfie' => $attendance->hasSelfie(),
        'selfie_url' => $attendance->getSelfieUrl(),
        'attendance_type' => $attendance->attendance_type
    ];
}));

// Initialize map when page loads
document.addEventListener('DOMContentLoaded', function() {
    initializeMap();
    addMarkersToMap();
    setupEventListeners();
});

function initializeMap() {
    // Default center (you can adjust this to your city/region)
    const defaultLat = attendanceData.length > 0 ? attendanceData[0].latitude || 40.7128 : 40.7128;
    const defaultLng = attendanceData.length > 0 ? attendanceData[0].longitude || -74.0060 : -74.0060;
    
    map = L.map('map').setView([defaultLat, defaultLng], 10);
    
    // Add OpenStreetMap tiles
    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
        attribution: '© OpenStreetMap contributors',
        maxZoom: 19
    }).addTo(map);
}

function addMarkersToMap() {
    attendanceData.forEach(function(attendance) {
        if (attendance.latitude && attendance.longitude) {
            // Create custom icon based on status
            let iconColor = 'red';
            switch(attendance.status) {
                case 'present': iconColor = 'green'; break;
                case 'late': iconColor = 'orange'; break;
                case 'absent': iconColor = 'red'; break;
                default: iconColor = 'red';
            }
            
            // Create marker
            const marker = L.marker([attendance.latitude, attendance.longitude])
                .addTo(map)
                .bindPopup(createPopupContent(attendance))
                .on('click', function() {
                    highlightAttendanceItem(attendance.id);
                });
            
            markers.push({
                marker: marker,
                id: attendance.id,
                data: attendance
            });
        }
    });
    
    // Fit map to show all markers
    if (markers.length > 0) {
        const group = new L.featureGroup(markers.map(m => m.marker));
        map.fitBounds(group.getBounds().pad(0.1));
    }
}

function createPopupContent(attendance) {
    return `
        <div class="text-sm">
            <div class="font-semibold text-lg mb-2">${attendance.user_name}</div>
            <div class="mb-1"><strong>Date:</strong> ${attendance.date}</div>
            <div class="mb-1"><strong>Time:</strong> ${attendance.time}</div>
            <div class="mb-1"><strong>Status:</strong> 
                <span class="px-2 py-1 rounded text-xs ${
                    attendance.status === 'present' ? 'bg-green-100 text-green-800' :
                    attendance.status === 'late' ? 'bg-yellow-100 text-yellow-800' :
                    attendance.status === 'absent' ? 'bg-red-100 text-red-800' :
                    'bg-gray-100 text-gray-800'
                }">${attendance.status.charAt(0).toUpperCase() + attendance.status.slice(1)}</span>
            </div>
            ${attendance.location_address ? `<div class="mb-2"><strong>Location:</strong> ${attendance.location_address}</div>` : ''}
            ${attendance.has_selfie ? '<div class="mt-2"><button onclick="showAttendanceDetails(' + attendance.id + ')" class="bg-red-500 text-white px-3 py-1 rounded text-xs hover:bg-red-600">View Details</button></div>' : ''}
        </div>
    `;
}

function setupEventListeners() {
    // Attendance item click handlers
    document.querySelectorAll('.attendance-item').forEach(function(item) {
        item.addEventListener('click', function() {
            const lat = parseFloat(this.dataset.lat);
            const lng = parseFloat(this.dataset.lng);
            const id = parseInt(this.dataset.id);
            
            if (lat && lng) {
                // Center map on this location
                map.setView([lat, lng], 15);
                
                // Find and open the corresponding marker popup
                const markerObj = markers.find(m => m.id === id);
                if (markerObj) {
                    markerObj.marker.openPopup();
                }
            }
            
            // Highlight this item
            highlightAttendanceItem(id);
        });
    });
    
    // Modal close handlers
    document.getElementById('closeModal').addEventListener('click', closeModal);
    document.getElementById('selfieModal').addEventListener('click', function(e) {
        if (e.target === this) {
            closeModal();
        }
    });
}

function highlightAttendanceItem(id) {
    // Remove previous highlights
    document.querySelectorAll('.attendance-item').forEach(function(item) {
        item.classList.remove('ring-2', 'ring-red-500');
    });
    
    // Add highlight to selected item
    const selectedItem = document.querySelector(`[data-id="${id}"]`);
    if (selectedItem) {
        selectedItem.classList.add('ring-2', 'ring-red-500');
        selectedItem.scrollIntoView({ behavior: 'smooth', block: 'center' });
    }
}

function showAttendanceDetails(attendanceId) {
    // Show loading
    document.getElementById('modalContent').innerHTML = '<div class="text-center py-8"><i class="fas fa-spinner fa-spin text-2xl text-gray-400"></i></div>';
    document.getElementById('selfieModal').classList.remove('hidden');
    
    // Fetch attendance details
    fetch(`/attendance/${attendanceId}`)
        .then(response => response.text())
        .then(html => {
            document.getElementById('modalContent').innerHTML = html;
        })
        .catch(error => {
            console.error('Error loading attendance details:', error);
            document.getElementById('modalContent').innerHTML = '<div class="text-center py-8 text-red-400">Error loading details</div>';
        });
}

function closeModal() {
    document.getElementById('selfieModal').classList.add('hidden');
}
</script>
@endpush