<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Payroll extends Model
{
    protected $fillable = [
        'user_id',
        'payroll_type',
        'batch_id',
        'year',
        'month',
        'days_present',
        'days_absent',
        'total_work_days',
        'monthly_compensation',
        'basic_salary',
        'hra',
        'conveyance',
        'special_allowance',
        'gross_salary',
        'employer_pf',
        'employer_esi',
        'ctc',
        'employee_pf',
        'employee_esi',
        'professional_tax',
        'net_salary',
        'daily_rate',
        'paid_leave_days',
        'special_deduction',
        'special_deduction_note',
        'reimbursement_total',
        'total_payout',
    ];

    protected $casts = [
        'monthly_compensation' => 'decimal:2',
        'basic_salary' => 'decimal:2',
        'hra' => 'decimal:2',
        'conveyance' => 'decimal:2',
        'special_allowance' => 'decimal:2',
        'gross_salary' => 'decimal:2',
        'employer_pf' => 'decimal:2',
        'employer_esi' => 'decimal:2',
        'ctc' => 'decimal:2',
        'employee_pf' => 'decimal:2',
        'employee_esi' => 'decimal:2',
        'professional_tax' => 'decimal:2',
        'net_salary' => 'decimal:2',
        'daily_rate' => 'decimal:2',
        'paid_leave_days' => 'decimal:2',
        'special_deduction' => 'decimal:2',
        'reimbursement_total' => 'decimal:2',
        'total_payout' => 'decimal:2',
    ];

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function reimbursements()
    {
        return $this->hasMany(Reimbursement::class);
    }

    public function calculatePayout()
    {
        // Use net_salary if available, otherwise fall back to monthly_compensation (legacy)
        $baseSalary = $this->net_salary > 0 ? $this->net_salary : $this->monthly_compensation;

        if ($this->total_work_days > 0) {
            $this->daily_rate = $baseSalary / $this->total_work_days;
        } else {
            $this->daily_rate = 0;
        }

        // Payout = (Days Present + Paid Leave Days) * Daily Rate
        $this->total_payout = $this->daily_rate * ($this->days_present + ($this->paid_leave_days ?? 0));
        
        // Add Reimbursements
        $this->total_payout += ($this->reimbursement_total ?? 0);

        // Subtract Special Deduction
        $this->total_payout -= ($this->special_deduction ?? 0);

        // Ensure payout is not negative
        $this->total_payout = max(0, $this->total_payout);

        // Rounding to 2 decimal places to avoid float precision issues
        $this->total_payout = round($this->total_payout, 2);

        return $this->total_payout;
    }

    public function scopeForMonth($query, $year, $month)
    {
        return $query->where('year', $year)->where('month', $month);
    }
}
