<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\StaffProfile;
use App\Models\Department;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;

class StaffController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $this->authorize('viewAny', User::class);
        
        $staff = User::role('Staff')
                    ->with(['staffProfile', 'staffProfile.department'])
                    ->latest()
                    ->paginate(15);
        
        $departments = Department::all();
        
        return view('staff.index', compact('staff', 'departments'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $this->authorize('create', User::class);
        
        $departments = Department::all();
        $supervisors = User::where('id', '!=', Auth::id())->get();
        
        return view('staff.create', compact('departments', 'supervisors'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $this->authorize('create', User::class);
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'attendance_type' => 'required|in:automatic,check-in',
            'designation' => 'nullable|string|max:255',
            'reports_to' => 'nullable|exists:users,id',
            'employee_id' => 'required|string|max:50|unique:staff_profiles',
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'department_id' => 'required|exists:departments,id',
            'position' => 'required|string|max:100',
            'salary' => 'required|numeric|min:0',
            'hire_date' => 'required|date',
            'work_days_per_week' => 'required|integer|min:1|max:7',
            'bank_name' => 'nullable|string|max:255',
            'bank_account_no' => 'nullable|string|max:50',
            'ifsc_code' => 'nullable|string|max:20',
        ]);
        
        $user = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'attendance_type' => $validated['attendance_type'],
            'designation' => $validated['designation'],
            'reports_to' => $validated['reports_to'],
        ]);
        
        $user->assignRole('Staff');
        
        StaffProfile::create([
            'user_id' => $user->id,
            'employee_id' => $validated['employee_id'],
            'phone' => $validated['phone'],
            'address' => $validated['address'],
            'department_id' => $validated['department_id'],
            'position' => $validated['position'],
            'salary' => $validated['salary'],
            'monthly_compensation' => $validated['salary'],
            'hire_date' => $validated['hire_date'],
            'work_days_per_week' => $validated['work_days_per_week'],
            'bank_name' => $validated['bank_name'] ?? null,
            'bank_account_no' => $validated['bank_account_no'] ?? null,
            'ifsc_code' => $validated['ifsc_code'] ?? null,
        ]);
        
        return redirect()->route('staff.index')
                        ->with('success', 'Staff member created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(User $staff)
    {
        $this->authorize('view', $staff);
        
        $staff->load(['staffProfile.department', 'tasks', 'attendances' => function($query) {
            $query->latest()->limit(10);
        }]);
        
        return view('staff.show', compact('staff'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(User $staff)
    {
        $this->authorize('update', $staff);
        
        $staff->load('staffProfile');
        $departments = Department::all();
        $supervisors = User::where('id', '!=', $staff->id)->get();
        
        return view('staff.edit', compact('staff', 'departments', 'supervisors'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, User $staff)
    {
        $this->authorize('update', $staff);
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => ['required', 'string', 'email', 'max:255', Rule::unique('users')->ignore($staff->id)],
            'employee_id' => ['required', 'string', 'max:50', Rule::unique('staff_profiles')->ignore($staff->staffProfile->id ?? null)],
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'department_id' => 'required|exists:departments,id',
            'position' => 'required|string|max:100',
            'salary' => 'required|numeric|min:0',
            'hire_date' => 'required|date',
            'attendance_type' => 'required|in:automatic,check-in',
            'designation' => 'nullable|string|max:255',
            'reports_to' => 'nullable|exists:users,id',
            'basic_salary' => 'nullable|numeric|min:0',
            'hra' => 'nullable|numeric|min:0',
            'conveyance' => 'nullable|numeric|min:0',
            'special_allowance' => 'nullable|numeric|min:0',
            'employer_pf' => 'nullable|numeric|min:0',
            'employer_esi' => 'nullable|numeric|min:0',
            'employee_pf' => 'nullable|numeric|min:0',
            'employee_esi' => 'nullable|numeric|min:0',
            'professional_tax' => 'nullable|numeric|min:0',
            'bank_name' => 'nullable|string|max:255',
            'bank_account_no' => 'nullable|string|max:50',
            'ifsc_code' => 'nullable|string|max:20',
        ]);
        
        $staff->update([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'attendance_type' => $validated['attendance_type'],
            'designation' => $validated['designation'],
            'reports_to' => $validated['reports_to'],
        ]);
        
        $staff->staffProfile->update([
            'employee_id' => $validated['employee_id'],
            'phone' => $validated['phone'],
            'address' => $validated['address'],
            'department_id' => $validated['department_id'],
            'position' => $validated['position'],
            'salary' => $validated['salary'],
            'hire_date' => $validated['hire_date'],
            'basic_salary' => $validated['basic_salary'] ?? 0,
            'hra' => $validated['hra'] ?? 0,
            'conveyance' => $validated['conveyance'] ?? 0,
            'special_allowance' => $validated['special_allowance'] ?? 0,
            'employer_pf' => $validated['employer_pf'] ?? 0,
            'employer_esi' => $validated['employer_esi'] ?? 0,
            'employee_pf' => $validated['employee_pf'] ?? 0,
            'employee_esi' => $validated['employee_esi'] ?? 0,
            'professional_tax' => $validated['professional_tax'] ?? 0,
            'bank_name' => $validated['bank_name'] ?? null,
            'bank_account_no' => $validated['bank_account_no'] ?? null,
            'ifsc_code' => $validated['ifsc_code'] ?? null,
        ]);
        
        return redirect()->route('staff.show', $staff)
                        ->with('success', 'Staff member updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(User $staff)
    {
        $this->authorize('delete', $staff);
        
        $staff->delete();
        
        return redirect()->route('staff.index')
                        ->with('success', 'Staff member deleted successfully.');
    }

    /**
     * Download sample CSV for bulk import.
     */
    public function downloadSample()
    {
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="staff_import_sample.csv"',
        ];

        $columns = [
            'Name', 
            'Email', 
            'Password', 
            'Attendance Type (automatic/check-in)', 
            'Employee ID', 
            'Department ID', 
            'Position', 
            'Salary (CTC)', 
            'Hire Date (YYYY-MM-DD)', 
            'Work Days Per Week (1-7)',
            'Phone',
            'Address',
            'Designation',
            'Reports To ID (Optional)',
            'Basic Salary',
            'HRA',
            'Conveyance',
            'Special Allowance',
            'Employer PF',
            'Employer ESI',
            'Employee PF',
            'Employee ESI',
            'Professional Tax',
            'Bank Name',
            'Bank Account No',
            'IFSC Code'
        ];

        $callback = function() use ($columns) {
            $file = fopen('php://output', 'w');
            fputcsv($file, $columns);
            
            // Add a sample row
            fputcsv($file, [
                'John Doe',
                'john.doe@example.com',
                'password123',
                'automatic',
                'EMP001',
                '1', // Assuming department 1 exists
                'Developer',
                '50000',
                date('Y-m-d'),
                '5',
                '1234567890',
                '123 Main St',
                'Senior Developer',
                '',
                '25000', // Basic
                '10000', // HRA
                '5000',  // Conveyance
                '5000',  // Special
                '1800',  // Employer PF
                '0',     // Employer ESI
                '1800',  // Employee PF
                '0',     // Employee ESI
                '200',   // PT
                'HDFC Bank',
                '123456789012',
                'HDFC0001234'
            ]);
            
            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    /**
     * Bulk import staff from CSV.
     */
    public function import(Request $request)
    {
        $this->authorize('create', User::class);

        $request->validate([
            'file' => 'required|file|mimes:csv,txt|max:2048',
        ]);

        $file = $request->file('file');
        $handle = fopen($file->getPathname(), 'r');
        
        // Skip header
        fgetcsv($handle);
        
        $row = 1;
        $successCount = 0;
        $errors = [];

        while (($data = fgetcsv($handle, 1000, ',')) !== FALSE) {
            $row++;
            
            // Basic validation of column count
            if (count($data) < 10) {
                $errors[] = "Row $row: Insufficient data.";
                continue;
            }

            try {
                // Map CSV columns to variables
                $name = $data[0];
                $email = $data[1];
                $password = $data[2];
                $attendanceType = $data[3];
                $employeeId = $data[4];
                $departmentId = $data[5];
                $position = $data[6];
                $salary = $data[7];
                $hireDate = $data[8];
                $workDays = $data[9];
                $phone = $data[10] ?? null;
                $address = $data[11] ?? null;
                $designation = $data[12] ?? null;
                $reportsTo = $data[13] ?? null;
                
                // Salary components
                $basicSalary = $data[14] ?? 0;
                $hra = $data[15] ?? 0;
                $conveyance = $data[16] ?? 0;
                $specialAllowance = $data[17] ?? 0;
                $employerPf = $data[18] ?? 0;
                $employerEsi = $data[19] ?? 0;
                $employeePf = $data[20] ?? 0;
                $employeeEsi = $data[21] ?? 0;
                $professionalTax = $data[22] ?? 0;
                
                // Bank details
                $bankName = $data[23] ?? null;
                $bankAccountNo = $data[24] ?? null;
                $ifscCode = $data[25] ?? null;

                // Validate data
                $validator = \Illuminate\Support\Facades\Validator::make([
                    'name' => $name,
                    'email' => $email,
                    'password' => $password,
                    'attendance_type' => $attendanceType,
                    'employee_id' => $employeeId,
                    'department_id' => $departmentId,
                    'position' => $position,
                    'salary' => $salary,
                    'hire_date' => $hireDate,
                    'work_days_per_week' => $workDays,
                    'basic_salary' => $basicSalary,
                    'hra' => $hra,
                    'conveyance' => $conveyance,
                    'special_allowance' => $specialAllowance,
                ], [
                    'name' => 'required|string|max:255',
                    'email' => 'required|string|email|max:255|unique:users',
                    'password' => 'required|string|min:8',
                    'attendance_type' => 'required|in:automatic,check-in',
                    'employee_id' => 'required|string|max:50|unique:staff_profiles',
                    'department_id' => 'required|exists:departments,id',
                    'position' => 'required|string|max:100',
                    'salary' => 'required|numeric|min:0',
                    'hire_date' => 'required|date',
                    'work_days_per_week' => 'required|integer|min:1|max:7',
                    'basic_salary' => 'nullable|numeric|min:0',
                    'hra' => 'nullable|numeric|min:0',
                    'conveyance' => 'nullable|numeric|min:0',
                    'special_allowance' => 'nullable|numeric|min:0',
                ]);

                if ($validator->fails()) {
                    $errors[] = "Row $row: " . implode(', ', $validator->errors()->all());
                    continue;
                }

                \Illuminate\Support\Facades\DB::transaction(function () use (
                    $name, $email, $password, $attendanceType, $designation, $reportsTo,
                    $employeeId, $phone, $address, $departmentId, $position, $salary, $hireDate, $workDays,
                    $basicSalary, $hra, $conveyance, $specialAllowance, $employerPf, $employerEsi,
                    $employeePf, $employeeEsi, $professionalTax, $bankName, $bankAccountNo, $ifscCode
                ) {
                    $user = User::create([
                        'name' => $name,
                        'email' => $email,
                        'password' => Hash::make($password),
                        'attendance_type' => $attendanceType,
                        'designation' => $designation,
                        'reports_to' => (is_numeric($reportsTo) && $reportsTo > 0) ? $reportsTo : null,
                    ]);
                    
                    $user->assignRole('Staff');
                    
                    StaffProfile::create([
                        'user_id' => $user->id,
                        'employee_id' => $employeeId,
                        'phone' => $phone,
                        'address' => $address,
                        'department_id' => $departmentId,
                        'position' => $position,
                        'salary' => $salary,
                        'monthly_compensation' => $salary,
                        'hire_date' => $hireDate,
                        'work_days_per_week' => $workDays,
                        'basic_salary' => $basicSalary,
                        'hra' => $hra,
                        'conveyance' => $conveyance,
                        'special_allowance' => $specialAllowance,
                        'employer_pf' => $employerPf,
                        'employer_esi' => $employerEsi,
                        'employee_pf' => $employeePf,
                        'employee_esi' => $employeeEsi,
                        'professional_tax' => $professionalTax,
                        'bank_name' => $bankName,
                        'bank_account_no' => $bankAccountNo,
                        'ifsc_code' => $ifscCode,
                    ]);
                });

                $successCount++;

            } catch (\Exception $e) {
                $errors[] = "Row $row: " . $e->getMessage();
            }
        }
        
        fclose($handle);

        $message = "Imported $successCount staff members successfully.";
        
        if (count($errors) > 0) {
            // Flash errors to session to be displayed in view
            return redirect()->route('staff.index')
                            ->with('success', $message)
                            ->with('import_errors', $errors);
        }

        return redirect()->route('staff.index')->with('success', $message);
    }
    
    /**
     * Show staff profile (for current user)
     */
    public function profile()
    {
        $user = Auth::user();
        $user->load(['staffProfile.department', 'tasks' => function($query) {
            $query->latest()->limit(5);
        }, 'attendances' => function($query) {
            $query->latest()->limit(10);
        }]);
        
        return view('staff.profile', compact('user'));
    }
    
    /**
     * Update staff profile (for current user)
     */
    public function updateProfile(Request $request)
    {
        $user = Auth::user();
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => ['required', 'string', 'email', 'max:255', Rule::unique('users')->ignore($user->id)],
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'password' => 'nullable|string|min:8|confirmed',
        ]);
        
        $updateData = [
            'name' => $validated['name'],
            'email' => $validated['email'],
        ];
        
        if (!empty($validated['password'])) {
            $updateData['password'] = Hash::make($validated['password']);
        }
        
        $user->update($updateData);
        
        if ($user->staffProfile) {
            $user->staffProfile->update([
                'phone' => $validated['phone'],
                'address' => $validated['address'],
            ]);
        }
        
        return back()->with('success', 'Profile updated successfully.');
    }

    // API Methods for Mobile App
    
    /**
     * API: Get staff list (Admin only)
     */
    public function apiIndex(Request $request)
    {
        $this->authorize('viewAny', User::class);
        
        $staff = User::role('Staff')
                    ->with(['staffProfile', 'staffProfile.department'])
                    ->latest()
                    ->paginate(15);
        
        return response()->json($staff);
    }

    /**
     * API: Create staff (Admin only)
     */
    public function apiStore(Request $request)
    {
        $this->authorize('create', User::class);
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8',
            'attendance_type' => 'required|in:automatic,check-in',
            'employee_id' => 'required|string|max:50|unique:staff_profiles',
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'department_id' => 'required|exists:departments,id',
            'position' => 'required|string|max:100',
            'salary' => 'required|numeric|min:0',
            'hire_date' => 'required|date',
            'work_days_per_week' => 'required|integer|min:1|max:7',
        ]);
        
        $user = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'attendance_type' => $validated['attendance_type'],
        ]);
        
        $user->assignRole('Staff');
        
        StaffProfile::create([
            'user_id' => $user->id,
            'employee_id' => $validated['employee_id'],
            'phone' => $validated['phone'],
            'address' => $validated['address'],
            'department_id' => $validated['department_id'],
            'position' => $validated['position'],
            'salary' => $validated['salary'],
            'monthly_compensation' => $validated['salary'],
            'hire_date' => $validated['hire_date'],
            'work_days_per_week' => $validated['work_days_per_week'],
        ]);
        
        return response()->json($user->load(['staffProfile', 'staffProfile.department']), 201);
    }

    /**
     * API: Show staff details
     */
    public function apiShow(Request $request, $id)
    {
        $staff = User::role('Staff')->findOrFail($id);
        $this->authorize('view', $staff);
        
        $staff->load(['staffProfile.department', 'tasks', 'attendances' => function($query) {
            $query->latest()->limit(10);
        }]);
        
        return response()->json($staff);
    }

    /**
     * API: Update staff (Admin only)
     */
    public function apiUpdate(Request $request, $id)
    {
        $staff = User::role('Staff')->findOrFail($id);
        $this->authorize('update', $staff);
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => ['required', 'string', 'email', 'max:255', Rule::unique('users')->ignore($staff->id)],
            'employee_id' => ['required', 'string', 'max:50', Rule::unique('staff_profiles')->ignore($staff->staffProfile->id ?? null)],
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'department_id' => 'required|exists:departments,id',
            'position' => 'required|string|max:100',
            'salary' => 'required|numeric|min:0',
            'hire_date' => 'required|date',
            'attendance_type' => 'required|in:automatic,check-in',
        ]);
        
        $staff->update([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'attendance_type' => $validated['attendance_type'],
        ]);
        
        $staff->staffProfile->update([
            'employee_id' => $validated['employee_id'],
            'phone' => $validated['phone'],
            'address' => $validated['address'],
            'department_id' => $validated['department_id'],
            'position' => $validated['position'],
            'salary' => $validated['salary'],
            'hire_date' => $validated['hire_date'],
        ]);
        
        return response()->json($staff->fresh()->load(['staffProfile', 'staffProfile.department']));
    }

    /**
     * API: Delete staff (Admin only)
     */
    public function apiDestroy(Request $request, $id)
    {
        $staff = User::role('Staff')->findOrFail($id);
        $this->authorize('delete', $staff);
        
        $staff->delete();
        
        return response()->json(['message' => 'Staff member deleted successfully']);
    }

    /**
     * API: Get staff list for task assignment (accessible to all authenticated users)
     * This endpoint allows staff to see other staff members for task reassignment
     */
    public function apiStaffForAssignment(Request $request)
    {
        // Get all staff members with basic info needed for task assignment
        $staff = User::role('Staff')
                    ->with(['staffProfile:user_id,department_id,position', 'staffProfile.department:id,name'])
                    ->select('id', 'name', 'email')
                    ->get();
        
        return response()->json($staff);
    }
}
