<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Department;
use App\Models\Task;
use App\Models\Attendance;
use App\Models\Payroll;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class DashboardController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        
        if ($user->hasRole('Admin')) {
            return $this->adminDashboard();
        } else {
            return $this->staffDashboard();
        }
    }

    private function adminDashboard()
    {
        $stats = [
            'total_staff' => User::role('Staff')->count(),
            'total_departments' => Department::count(),
            'pending_tasks' => Task::where('status', 'pending')->count(),
            'in_progress_tasks' => Task::where('status', 'in_progress')->count(),
            'completed_tasks' => Task::where('status', 'completed')->count(),
            'overdue_tasks' => Task::overdue()->count(),
            'due_soon_tasks' => Task::dueSoon(3)->count(),
            'present_today' => Attendance::where('date', Carbon::today())
                                       ->where('status', 'present')
                                       ->count(),
        ];

        $recent_tasks = Task::with(['creator', 'currentDepartment', 'currentAssignee'])
                           ->latest()
                           ->take(5)
                           ->get();

        $departments = Department::withCount('staffProfiles')->get();

        return view('dashboard.admin', compact('stats', 'recent_tasks', 'departments'));
    }

    private function staffDashboard()
    {
        $user = Auth::user();
        $staffProfile = $user->staffProfile;
        
        $stats = [
            'assigned_tasks' => Task::where('current_assigned_to', $user->id)
                                  ->where('status', '!=', 'completed')
                                  ->count(),
            'completed_tasks' => Task::where('current_assigned_to', $user->id)
                                   ->where('status', 'completed')
                                   ->count(),
            'attendance_this_month' => Attendance::where('user_id', $user->id)
                                                ->whereMonth('date', Carbon::now()->month)
                                                ->where('status', 'present')
                                                ->count(),
            'department' => $staffProfile ? $staffProfile->department->name : 'Not Assigned',
        ];

        $my_tasks = Task::where('current_assigned_to', $user->id)
                       ->with(['creator', 'currentDepartment'])
                       ->latest()
                       ->take(5)
                       ->get();

        $recent_attendance = Attendance::where('user_id', $user->id)
                                      ->latest('date')
                                      ->take(7)
                                      ->get();

        return view('dashboard.staff', compact('stats', 'my_tasks', 'recent_attendance'));
    }

    /**
     * API method for dashboard data
     */
    public function apiIndex(Request $request)
    {
        $user = $request->user();
        
        if ($user->hasRole('Admin')) {
            return $this->apiAdminDashboard();
        } else {
            return $this->apiStaffDashboard($user);
        }
    }

    private function apiAdminDashboard()
    {
        $stats = [
            'total_staff' => User::role('Staff')->count(),
            'total_departments' => Department::count(),
            'pending_tasks' => Task::where('status', 'pending')->count(),
            'in_progress_tasks' => Task::where('status', 'in_progress')->count(),
            'completed_tasks' => Task::where('status', 'completed')->count(),
            'present_today' => Attendance::where('date', Carbon::today())
                                       ->where('status', 'present')
                                       ->count(),
        ];

        $recent_tasks = Task::with(['creator', 'currentDepartment', 'currentAssignee'])
                           ->latest()
                           ->take(5)
                           ->get();

        $departments = Department::withCount('staffProfiles')->get();

        return response()->json([
            'stats' => $stats,
            'recent_tasks' => $recent_tasks,
            'departments' => $departments
        ]);
    }

    private function apiStaffDashboard($user)
    {
        $staffProfile = $user->staffProfile;
        
        $stats = [
            'assigned_tasks' => Task::where('current_assigned_to', $user->id)
                                  ->where('status', '!=', 'completed')
                                  ->count(),
            'completed_tasks' => Task::where('current_assigned_to', $user->id)
                                   ->where('status', 'completed')
                                   ->count(),
            'attendance_this_month' => Attendance::where('user_id', $user->id)
                                                ->whereMonth('date', Carbon::now()->month)
                                                ->where('status', 'present')
                                                ->count(),
            'department' => $staffProfile ? $staffProfile->department->name : 'Not Assigned',
        ];

        $my_tasks = Task::where('current_assigned_to', $user->id)
                       ->with(['creator', 'currentDepartment'])
                       ->latest()
                       ->take(5)
                       ->get();

        $recent_attendance = Attendance::where('user_id', $user->id)
                                      ->latest('date')
                                      ->take(7)
                                      ->get();

        return response()->json([
            'stats' => $stats,
            'my_tasks' => $my_tasks,
            'recent_attendance' => $recent_attendance
        ]);
    }
}
