<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Reimbursement;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class ReimbursementController extends Controller
{
    public function index(Request $request)
    {
        $user = $request->user();
        
        $query = Reimbursement::with('user:id,name,email');

        if ($user->isStaff()) {
            $query->where('user_id', $user->id);
        }

        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        $reimbursements = $query->latest()->paginate(20);

        return response()->json([
            'success' => true,
            'data' => $reimbursements
        ]);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'amount' => 'required|numeric|min:0.01',
            'description' => 'required|string',
            'image' => 'required|image|max:10240', // 10MB max
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $path = $request->file('image')->store('reimbursements', 'public');

        $reimbursement = Reimbursement::create([
            'user_id' => $request->user()->id,
            'amount' => $request->amount,
            'description' => $request->description,
            'image_path' => $path,
            'status' => 'pending',
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Reimbursement request submitted successfully',
            'data' => $reimbursement
        ], 201);
    }

    public function updateStatus(Request $request, Reimbursement $reimbursement)
    {
        // Ensure user is admin
        if (!$request->user()->isAdmin()) {
            return response()->json(['success' => false, 'message' => 'Unauthorized'], 403);
        }

        $validator = Validator::make($request->all(), [
            'status' => 'required|in:approved,rejected,pending',
            'admin_comment' => 'nullable|string',
            'approved_amount' => 'nullable|numeric|min:0',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $data = [
            'status' => $request->status,
            'admin_comment' => $request->admin_comment,
        ];

        if ($request->status === 'approved') {
            // Default to requested amount if not specified
            $data['approved_amount'] = $request->has('approved_amount') 
                ? $request->approved_amount 
                : $reimbursement->amount;
        }

        $reimbursement->update($data);

        return response()->json([
            'success' => true,
            'message' => 'Reimbursement status updated',
            'data' => $reimbursement
        ]);
    }
}
